<?php
// upload-convert.php — accepts:
//  A) multipart/form-data field: image
//  B) raw binary body (Content-Type: image/png|image/jpeg)
//  C) JSON: { "image_base64": "...", "fileName": "...", "mimeType": "image/png" }
// Converts to WebP and returns { success, url }.

header('Content-Type: application/json');

function fail($msg, $extra = []) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => $msg] + $extra);
    exit;
}

function save_webp($srcTmpPath, $origName) {
    $outDir = __DIR__ . '/converted';
    if (!is_dir($outDir)) mkdir($outDir, 0777, true);

    $info = getimagesize($srcTmpPath);
    if (!$info) fail('Invalid image file');

    switch ($info['mime']) {
        case 'image/png':
            $img = imagecreatefrompng($srcTmpPath);
            break;
        case 'image/jpeg':
            $img = imagecreatefromjpeg($srcTmpPath);
            break;
        default:
            fail('Unsupported file type', ['mime' => $info['mime']]);
    }

    $base = preg_replace('/[^A-Za-z0-9._-]/', '_', pathinfo($origName, PATHINFO_FILENAME)) ?: 'image';
    $outPath = $outDir . '/' . $base . '.webp';

    if (!function_exists('imagewebp')) fail('PHP GD with WebP not enabled');
    if (!imagewebp($img, $outPath, 80)) fail('Conversion failed');

    imagedestroy($img);

    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $url = $scheme . '://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['REQUEST_URI']) . '/converted/' . basename($outPath);

    echo json_encode(['success' => true, 'url' => $url, 'file' => basename($outPath)]);
    exit;
}

/* --------- C) JSON body --------- */
$ct = $_SERVER['CONTENT_TYPE'] ?? '';
if (stripos($ct, 'application/json') === 0) {
    $raw = file_get_contents('php://input');
    $j = json_decode($raw, true);
    if (!$j || empty($j['image_base64'])) fail('No image_base64 in JSON');
    $bin = base64_decode($j['image_base64'], true);
    if ($bin === false) fail('Invalid base64');
    $tmp = tempnam(sys_get_temp_dir(), 'img_');
    file_put_contents($tmp, $bin);
    save_webp($tmp, $j['fileName'] ?? 'image.png');
}

/* --------- A) multipart/form-data --------- */
if (!empty($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
    save_webp($_FILES['image']['tmp_name'], $_FILES['image']['name']);
}

/* --------- B) raw binary (image/*) --------- */
if (preg_match('#^image/(png|jpeg)$#i', $ct)) {
    $raw = file_get_contents('php://input');
    if ($raw === '' || $raw === false) fail('Empty raw body');
    $tmp = tempnam(sys_get_temp_dir(), 'img_');
    file_put_contents($tmp, $raw);
    $name = (stripos($ct, 'jpeg') !== false) ? 'image.jpg' : 'image.png';
    save_webp($tmp, $name);
}

/* --------- fallback --------- */
fail('No image uploaded. Send multipart field "image" or raw binary body with image/png or image/jpeg, or JSON with image_base64.');
